/*
 * xant.c
 */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>


#include <stdio.h>

/* 
 * The following should be placed in an "xant.h" file.
 */
#define XtNdebug "debug"
#define XtCDebug "Debug"
#define XtNtruchet "truchet"
#define XtCTruchet "Truchet"
#define XtNdelayTime "delayTime"
#define XtCdelayTime "DelayTime"
#define XtNtimestep "timestep"
#define XtCtimestep "Timestep"
#define XtNantNumber "antNumber"
#define XtCantNumber "AntNumber"
#define XtNcellSize "cellSize"
#define XtCcellSize "CellSize"
#define XtNuniverseSize "universeSize"
#define XtCuniverseSize "UniverseSize"
#define XtNuniverseHeight "universeHeight"
#define XtCuniverseHeight "UniverseHeight"
#define XtNuniverseWidth "universeWidth"
#define XtCuniverseWidth "UniverseWidth"

#define MAX_UNIV_HEIGHT	 1000
#define MAX_UNIV_WIDTH	 1000

	/* status codes (for runstat flag): */
#define CONTINUE         0
#define INTERRUPT        1
#define TRAP             2
#define STOP             3

unsigned char 	universe[MAX_UNIV_WIDTH][MAX_UNIV_HEIGHT];
unsigned char	numstates;
int		runstat;
int		rule[10];
long int	curr_step_number;
static char 	*direction[] = {"east ", "north", "west ", "south"};
struct	{
        int	x_coord;
	int	y_coord;
	int	heading;
	int	prevheading;
	int	x_prev;
	int	y_prev;
	int	min_x;
	int	min_y;
	int	max_x;
	int	max_y;
      }		ant;
int		xtrap, ytrap;





#define MINCELLSIZE  5
#define MAXCELLSIZE  50

Pixmap big_picture;		/* a copy of the whole picture */
Pixmap clear_stipple, truchet_stipple; /* for clearing the big_picture */
GC canv_gc;			/* a regular GC for drawing in big_pic(1-bit)*/
GC dash_gc;			/* a dashed  GC for drawing in big_pic(1-bit)*/
GC *draw_gc;			/* array of GCs for drawing in big_pic(1-bit)*/
GC copy_gc;			/* for copying pixmap->window, screen depth */

Widget viewport;
Widget canvas;			/* this is the drawing surface */

Widget  quitButton,		/* The control panel buttons */
        goButton, 
        stepButton,
        antDataLabel;
static char	timestep_str[32];
static	char antDataString[100] = "";
Pixmap 	antStatesBitmap;	/* a pixmap for a picture of all the states */

XtAppContext app_context;	/* the application context */


/* data structure for application resources */
typedef struct {
	Pixel copy_fg;
	Pixel copy_bg;
	int delay_time;
	int timestep;
	int univ_size;
	int univ_height;
	int univ_width;
	int cell_size;
	int ant_number;
	Boolean debug;
	Boolean truchet;
} AppData, *AppDataPtr;

AppData app_data;

/* resource list */
static XtResource resources[] = {
	{
		XtNforeground,
		XtCForeground,
		XtRPixel,
		sizeof(Pixel),
		XtOffsetOf(AppData, copy_fg),
		XtRString,
		XtDefaultForeground
	},
	{
		XtNbackground,
		XtCBackground,
		XtRPixel,
		sizeof(Pixel),
		XtOffsetOf(AppData, copy_bg),
		XtRString,
		XtDefaultBackground
	},
	{
		XtNdelayTime,
		XtCdelayTime,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, delay_time),
		XtRImmediate,
		(XtPointer) 2,
	},
	{
		XtNtimestep,
		XtCtimestep,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, timestep),
		XtRImmediate,
		(XtPointer) 100,
	},
	{
		XtNantNumber,
		XtCantNumber,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, ant_number),
		XtRImmediate,
		(XtPointer) 2,
	},
	{
		XtNcellSize,
		XtCcellSize,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, cell_size),
		XtRImmediate,
		(XtPointer) 17,
	},
	{
		XtNuniverseHeight,
		XtCuniverseHeight,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, univ_height),
		XtRImmediate,
		(XtPointer) 0,
	},
	{
		XtNuniverseWidth,
		XtCuniverseWidth,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, univ_width),
		XtRImmediate,
		(XtPointer) 0,
	},
	{
		XtNuniverseSize,
		XtCuniverseSize,
		XtRInt,
		sizeof(int),
		XtOffsetOf(AppData, univ_size),
		XtRImmediate,
		(XtPointer) 100,
	},
	{
		XtNdebug,
		XtCDebug,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, debug),
		XtRImmediate,
		(XtPointer) FALSE,
	},
	{
		XtNtruchet,
		XtCTruchet,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, truchet),
		XtRImmediate,
		(XtPointer) FALSE,
	},
};

/* Command-line options table */
static XrmOptionDescRec options[] = {
	{"-delay",         "*delayTime",           XrmoptionSepArg, NULL},
	{"-d",             "*delayTime",           XrmoptionSepArg, NULL},
	{"-timestep",      "*timestep",            XrmoptionSepArg, NULL},
	{"-step",          "*timestep",            XrmoptionSepArg, NULL},
	{"-t",             "*timestep",            XrmoptionSepArg, NULL},
	{"-cellsize",      "*cellSize",            XrmoptionSepArg, NULL},
	{"-csize",         "*cellSize",            XrmoptionSepArg, NULL},
	{"-height",        "*universeHeight",      XrmoptionSepArg, NULL},
	{"-h",             "*universeHeight",      XrmoptionSepArg, NULL},
	{"-width",         "*universeWidth",       XrmoptionSepArg, NULL},
	{"-w",             "*universeWidth",       XrmoptionSepArg, NULL},
	{"-size",          "*universeSize",        XrmoptionSepArg, NULL},
	{"-usize",         "*universeSize",        XrmoptionSepArg, NULL},
	{"-rule",          "*antNumber",           XrmoptionSepArg, NULL},
	{"-number",        "*antNumber",           XrmoptionSepArg, NULL},
	{"-ant",           "*antNumber",           XrmoptionSepArg, NULL},
	{"-debug",         "*debug",               XrmoptionNoArg, "True"},
	{"-truchet",       "*truchet",             XrmoptionNoArg, "True"},
	{"-fg",            "*foreground",          XrmoptionSepArg, NULL},
	{"-foreground",    "*foreground",          XrmoptionSepArg, NULL},
};
/*************************************************************************/
static void RedrawPicture(), DrawCell(), DrawCellInPixmap(), DrawAnt();
static void start_timer(), stop_timer();
static void ClearTheBigPicture(), RefreshTheBigPicture();
static void ToggleTruchet();
static void Go_Trap(), Pause(), GoTrap(), Quit(), DoStep(), DoManySteps();
/*************************************************************************/
/*      Procedures for the buttons                                       */
/*************************************************************************/
/* ARGSUSED */
static void 
PauseOrCont(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  switch(runstat) {
  case STOP:
    runstat=CONTINUE;
    XtVaSetValues(goButton,XtNlabel, "  Pause ", NULL); 
    start_timer();
    break;

  case CONTINUE:
  case TRAP:
    runstat=STOP;
    stop_timer();
    XtVaSetValues(goButton,XtNlabel, "Continue", NULL); 
    break;

  default:
    fprintf(stderr, "unknown runstat.\n");
    break;
  }
}
/* ARGSUSED */
static void 
Pause(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
     runstat=CONTINUE;		/* just lie, and say we were already running */
     PauseOrCont();		/* and pretent the Pause button was hit */
}

/* ARGSUSED */
static void 
Quit(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  exit();
}
/* ARGSUSED */
static void 
Step(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  if ((runstat != STOP)) Pause();
  DoStep();
}
static void
DoManySteps()
{
  int i;
  char *stepcount;
  int numsteps;
  
  if ((numsteps = atoi(timestep_str)) > 0) {
    app_data.timestep = numsteps;
  }
  
  if (app_data.timestep > 1) {
    for (i=1; i<app_data.timestep; i++) {
      step_ant();
    }
    RefreshTheBigPicture();
    DoStep();			/* one extra step to update buttons */
  }
}

static void 
DoStep()
{
  step_ant();
  if ((runstat == TRAP) && 
      (ant.x_coord == xtrap) && (ant.y_coord == ytrap)) {
    Pause();
  }
  if ((runstat == STOP) || 
      (app_data.delay_time >5) || (curr_step_number%10 == 0) ) {
    sprintf(antDataString, "Step %ld  Entering %d,%d  Turning %s ", 
	    curr_step_number, 
	    ant.x_coord, ant.y_coord, direction[ant.heading]);
    XtVaSetValues(antDataLabel,XtNlabel,antDataString,NULL);
  }
  DrawCell(ant.x_prev,ant.y_prev);
  DrawAnt(ant.x_coord,ant.y_coord, True);
}

static void 
GoTrap(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  runstat = TRAP;
  XtVaSetValues(goButton,XtNlabel, "  Pause ", NULL); 
  start_timer();
}


Oops() {	/* ant has hit the edge */
	printf("ant has hit the edge at step %d!\n", curr_step_number);
	ant.x_coord = ant.x_prev;
	ant.y_coord = ant.y_prev;
	if (runstat != STOP ) {
	  PauseOrCont();
	}
}

/*************************************************************************/
static XtIntervalId timer;

static void
start_timer()
{
  unsigned long interval;

  interval = app_data.delay_time;
  DoStep();
/*  XFlush(XtDisplay(canvas)); */
  if (runstat != STOP) 
    timer = XtAppAddTimeOut(app_context, interval, start_timer, NULL);
}

static void
stop_timer()
{
  if( timer ) {
    XtRemoveTimeOut(timer);
    timer = NULL;
  }
}

/*************************************************************************/

static void Syntax(argc, argv)
int argc;
char * argv[];
{
    int i;
    static int errs = False;

    /* first argument is program name - skip that */
    for (i = 1; i < argc; i++) {
      if (!errs++) /* do first time through */
	fprintf(stderr, "xant: command line option not understood:\n");
      fprintf(stderr, "option: %s\n", argv[i]);
    }

    fprintf(stderr, "xant understands all standard Xt commandline options.\n");

    fprintf(stderr, "Additional options are as follows:\n");
    fprintf(stderr, "Option             Valid Range\n");
    fprintf(stderr, "-rule              ant rule number\n");
    fprintf(stderr, "-delay             delay time (millisec)\n");
    fprintf(stderr, "-cellsize          MINCELLSIZE to MAXCELLSIZE\n");
    fprintf(stderr, "-height            2 to MAX_UNIV_HEIGHT (in cells)\n");
    fprintf(stderr, "-width             2 to MAX_UNIV_WIDTH  (in cells)\n");
    fprintf(stderr, "-size              if height or width not given\n");
    fprintf(stderr, "-fg                color name\n");
    fprintf(stderr, "-foreground        color name\n");
    fprintf(stderr, "-truchet           \n");
    fprintf(stderr, "-debug             \n");
}

/*************************************************************************/
/*            main routine                                               */
/*************************************************************************/
main(argc, argv)
     int argc;
     char *argv[];
{
  static char antNameString[12];

  Widget topLevel, vpane, buttonbox, buttonbox2, manyStepButton, timestepval;
  Widget antNameLabel, antStatesLabel;
  int	 i;

  /* translations for pane widget */
  String trans =
	"<Key>space:    Step()                  \n\
	 <Key>s:        Step()                  \n\
	 <Key>!:        DoManySteps()           \n\
	 <Key>g:        PauseOrCont()           \n\
	 <Key>p:        Pause()                 \n\
         <Key>r:        RefreshTheBigPicture()  \n\
         <Key>d:        ToggleTruchet()         \n\
	 Shift<Key>q:   Quit()                  \n\
         Ctrl<Key>c:    Pause()                 \n\
         Ctrl<Key>l:    RefreshTheBigPicture()  \n\
	 Ctrl<Key>q:    Quit()                  \n\
	 <Key>t:        GoTrap()               \n\
         ";
  /* additonal translations canvas widget */
  String canvas_trans =
	"<Expose>:	RedrawPicture()	        \n\
        ";
/*          <Btn1Down>:    DrawCell()              \n\
         <Btn2Down>:    UndrawCell()            \n\
         <Btn3Down>:    ToggleCell()            \n\
         <Btn1Motion>:  DrawCell()              \n\
         <Btn2Motion>:  UndrawCell()            \n\
         <Btn3Motion>:  ToggleCell()            \n\
        {"DrawCell", 	DrawCell},
        {"UndrawCell",	UndrawCell},
        {"ToggleCell",	ToggleCell},

*/
  static XtActionsRec window_actions[] = {
	{"RedrawPicture",	RedrawPicture},
	{"RefreshTheBigPicture", RefreshTheBigPicture},
	{"ToggleTruchet", ToggleTruchet},
	{"Step",        Step},
	{"DoManySteps", DoManySteps},
	{"Quit",        Quit},
	{"Pause",       Pause},
	{"GoTrap",      GoTrap},
	{"PauseOrCont", PauseOrCont},
    };

    topLevel = XtVaAppInitialize(
		 &app_context,       /* Application context */
		 "XAnt", 
		 options, XtNumber(options), 
		 &argc, argv,        /* command line args */
		 NULL,               /* for missing app-defaults file */
		 NULL);              /* terminate varargs list */

    /* XtInitialize leaves program name in args */
    if (argc > 1)
	Syntax(argc, argv);

    XtGetApplicationResources(topLevel, 
		&app_data, 
		resources, 
		XtNumber(resources), 
		NULL, 
		0);

    /*
     * We must check the application resource values here.
     * Otherwise, user could supply out of range values and crash program.
     * Conversion routines do this automatically, so colors arealready checked.
     */
  if ((app_data.cell_size<MINCELLSIZE) || (app_data.cell_size>MAXCELLSIZE)) {
    fprintf(stderr, "xant: cell size must be between %d and %d pixels\n", 
	    MINCELLSIZE, MAXCELLSIZE);
    exit(1);
  }
  if (app_data.univ_size < 2 ) {
       fprintf(stderr, "xant: universe size must greater than 1 cell.\n");
       exit(1);
     }
  if (app_data.univ_height < 2) app_data.univ_height = app_data.univ_size;
  if (app_data.univ_width < 2) app_data.univ_width = app_data.univ_size;
  if (app_data.univ_height> MAX_UNIV_HEIGHT) {
    fprintf(stderr, "xant: universe height must be less than %d cells\n", 
	    MAX_UNIV_HEIGHT);
    exit(1);
  }
  if (app_data.univ_height> MAX_UNIV_WIDTH) {
    fprintf(stderr, "xant: universe width must be less than %d cells\n", 
	    MAX_UNIV_WIDTH);
    exit(1);
  }

/* as a hack, ensure that the univ. size is divisble by 4, so the ant */
/* starts in an even cell */
  app_data.univ_width  = ((3+app_data.univ_width) /4)*4;
  app_data.univ_height = ((3+app_data.univ_height)/4)*4;

  if (app_data.debug) {
    fprintf(stderr, "Doing ant %d on a %dx%d universe\n", 
	    app_data.ant_number, app_data.univ_width, app_data.univ_height);
  }
    /* begin application code */

  Initialize();
  set_up_things(topLevel);
  
  sprintf(antNameString, "Ant %d", 
	  app_data.ant_number);
  sprintf(antDataString, "Step %ld   Entering %d,%d  Turning %s", 
	  curr_step_number, ant.x_coord, ant.y_coord, direction[ant.heading]);


/* the outer container for the main window */
  vpane = XtVaCreateManagedWidget("vpane", 
				  panedWidgetClass, topLevel,
			  XtNtranslations, XtParseTranslationTable(trans),
				  NULL);

/* Make the "control panel" */
  buttonbox = XtVaCreateManagedWidget("buttonbox", 
			       boxWidgetClass, vpane,
			       NULL); 

  antNameLabel = XtVaCreateManagedWidget("antName", 
					 labelWidgetClass, buttonbox, 
					 XtNlabel,   antNameString,
					 NULL, 0);

  antStatesLabel = XtVaCreateManagedWidget("antStates", 
					 labelWidgetClass, buttonbox, 
					 XtNbitmap,   antStatesBitmap,
					 NULL, 0);

  goButton = XtVaCreateManagedWidget("goButton",
				     commandWidgetClass, buttonbox,
				     XtNlabel,		"  Start ",
				     NULL); 
  XtAddCallback(goButton, XtNcallback, PauseOrCont, NULL);

  stepButton = XtVaCreateManagedWidget("stepButton", 
				       commandWidgetClass, buttonbox,
				       XtNlabel,"Single Step",
				       NULL); 
  XtAddCallback(stepButton, XtNcallback, Step, NULL);

  buttonbox2 = XtVaCreateManagedWidget("buttonbox2", 
			       boxWidgetClass, buttonbox,
			       XtNorientation, XtorientHorizontal,
			       NULL); 
  sprintf(timestep_str,"%d",app_data.timestep);
  timestepval = XtVaCreateManagedWidget("timestepval",
				   asciiTextWidgetClass, buttonbox2,
				   XtNstring, timestep_str,
				   XtNeditType, XawtextEdit,
				   XtNlength, 5,
				   XtNuseStringInPlace, True,
				   XtNwidth, 50,
				   NULL);
  manyStepButton =   quitButton = XtVaCreateManagedWidget("manyStepButton",
				       commandWidgetClass,   buttonbox2, 
				       XtNlabel,"steps",
				       NULL); 
  XtAddCallback(manyStepButton, XtNcallback, DoManySteps, NULL);


  quitButton = XtVaCreateManagedWidget("quitButton",
				       commandWidgetClass,   buttonbox, 
				       XtNlabel,"Quit",
				       NULL); 
  XtAddCallback(quitButton, XtNcallback, Quit, NULL);

  antDataLabel = XtVaCreateManagedWidget("antData", 
					 labelWidgetClass, buttonbox, 
					 XtNlabel,   antDataString,
					 NULL);
/* Now make the canvas */
  viewport = XtVaCreateManagedWidget("viewport", 
				     viewportWidgetClass, vpane, 
				     XtNallowVert,	True,
				     XtNallowHoriz,	True,
				     NULL); 

  canvas = XtVaCreateManagedWidget("canvas", widgetClass, viewport, 
	XtNtranslations, XtParseTranslationTable(canvas_trans),
			XtNwidth, app_data.univ_width*app_data.cell_size,
			XtNheight,app_data.univ_height*app_data.cell_size,
			NULL);

/* Realize widgets, and start main loop */
  XtAppAddActions(app_context, window_actions, XtNumber(window_actions));
  XtRealizeWidget(topLevel);

  ClearTheBigPicture();
  DrawAnt(ant.x_coord,ant.y_coord, False);

  XtAppMainLoop(app_context);
}

set_up_things(w)
Widget w;
{
  GC temp_gc;
  XGCValues values;
  int x, y;
  int i, j;
  Pixmap stipple;
  static unsigned char dashlist[2] = {1,1};


#define PATTERNSIZE 8
#define NUMPATTERNS 11
  static char patterns[NUMPATTERNS][PATTERNSIZE] = {
   {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* white */
   {0x11, 0x22, 0x11, 0x22, 0x11, 0x22, 0x11, 0x22}, /* grey+white | stripe */
   {0x00, 0x66, 0x66, 0x00, 0x00, 0x66, 0x66, 0x00}, /* spots */
   {0x89, 0x44, 0x22, 0x11, 0x88, 0x44, 0x22, 0x11}, /* lt. / stripe */ 
   {0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66}, /* | bars */
   {0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa}, /* 50% grey */
   {0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00}, /* - bars */
   {0xee, 0xdd, 0xbb, 0x77, 0xee, 0xdd, 0xbb, 0x76}, /* dark \ stripe */
   {0xff, 0x99, 0x99, 0xff, 0xff, 0x99, 0x99, 0xff}, /* spots */
   {0xaa, 0xff, 0xff, 0x55, 0xaa, 0xff, 0xff, 0x55}, /* black+grey - stripe */
   {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}  /* black */
 };

  big_picture = XCreatePixmap(XtDisplay(w),
			      RootWindowOfScreen(XtScreen(w)),
			      app_data.univ_height * app_data.cell_size,
			      app_data.univ_width * app_data.cell_size,
			      1);

  values.foreground = app_data.copy_fg;
  values.background = app_data.copy_bg;
  copy_gc = XCreateGC(XtDisplay(w), RootWindowOfScreen(XtScreen(w)),
		      GCForeground |GCBackground, &values);


  values.foreground = 1;
  values.background = 0;
  values.line_style = LineSolid; 
  canv_gc = XCreateGC(XtDisplay(w), big_picture,
		      GCForeground |GCBackground | GCLineStyle, &values);

 /* make the GCs for drawing the tiles */
  if (numstates > NUMPATTERNS) {
    fprintf(stderr,"WARNING: not enough patterns for all states\n");
  }

  if (numstates > 5) {
    antStatesBitmap = XCreatePixmap(XtDisplay(w),
				    RootWindowOfScreen(XtScreen(w)),
				    16*((numstates+1)/2), 32, 1);
  }else {
    antStatesBitmap = XCreatePixmap(XtDisplay(w),
				    RootWindowOfScreen(XtScreen(w)),
				    16*numstates, 16, 1);
  }
  values.foreground = 1;
  values.background = 0;
  values.line_style = LineSolid; 
  values.fill_style = FillOpaqueStippled;
  draw_gc = (GC *)XtMalloc(numstates*sizeof(GC));
  temp_gc = XCreateGC(XtDisplay(w), antStatesBitmap,
			  GCForeground |GCBackground |GCLineStyle |GCFillStyle,
			  &values);  
  XFillRectangle(XtDisplay(w), antStatesBitmap, temp_gc, 0,0,
		 (numstates>5)?16*((numstates+1)/2):16*numstates,
		 (numstates>5)? 32                 :16);

  for (i=0; i<numstates; i++) {
    j= (i*NUMPATTERNS)/numstates;
    if (i==numstates-1) j=NUMPATTERNS-1; /* be sure we get the last one */
    draw_gc[i]= XCreateGC(XtDisplay(w), big_picture,
			  GCForeground |GCBackground |GCLineStyle |GCFillStyle,
			  &values); 
    stipple=XCreateBitmapFromData(XtDisplay(w), big_picture, 
				  patterns[j], PATTERNSIZE, PATTERNSIZE);
    XSetStipple(XtDisplay(w), draw_gc[i], stipple);

    XSetStipple(XtDisplay(w), temp_gc, stipple); 
    if (numstates > 5) 
       XFillRectangle(XtDisplay(w), antStatesBitmap, temp_gc,
		      16*(i/2), (i%2)?17:0, 16,16);
    else
       XFillRectangle(XtDisplay(w), antStatesBitmap, temp_gc,
		      16*i, 0, 16,16);
  }
  XFreeGC(XtDisplay(w), temp_gc);

  values.foreground = 1;
  values.background = 0;
  values.line_style = LineOnOffDash; 
  dash_gc = XCreateGC(XtDisplay(w), big_picture,
		      GCForeground |GCBackground | GCLineStyle, &values);
  XSetDashes(XtDisplay(w), dash_gc, 2, dashlist, 2);

  /* create some stipples for clearing the big picture */
  values.foreground = 0;
  values.background = 1;
  temp_gc = XCreateGC(XtDisplay(w), big_picture, 
		      GCForeground |GCBackground, &values);
  
  clear_stipple = XCreatePixmap(XtDisplay(w),
			      RootWindowOfScreen(XtScreen(w)),
			      app_data.cell_size,
			      app_data.cell_size,
			      1);
  XFillRectangle(XtDisplay(w), clear_stipple, temp_gc, 0, 0, 
		 app_data.cell_size, app_data.cell_size);
  truchet_stipple = XCreatePixmap(XtDisplay(w),
			      RootWindowOfScreen(XtScreen(w)),
			      2*app_data.cell_size,
			      2*app_data.cell_size,
			      1);
  XFillRectangle(XtDisplay(w), truchet_stipple, temp_gc, 0, 0, 
		 2*app_data.cell_size, 2*app_data.cell_size);

  XDrawLine(XtDisplay(w), clear_stipple, dash_gc, 0, 0, 
	    app_data.cell_size-1, 0);
  XDrawLine(XtDisplay(w), clear_stipple, dash_gc, 0, 0, 
	    0, app_data.cell_size-1);

  XDrawLine(XtDisplay(w), truchet_stipple, dash_gc, 0, 0, 
	    0, 2*app_data.cell_size-1);
  XDrawLine(XtDisplay(w), truchet_stipple, dash_gc, 0, 0, 
	    2*app_data.cell_size-1, 0);
  XDrawLine(XtDisplay(w), truchet_stipple, dash_gc, app_data.cell_size, 0,
	    app_data.cell_size, 2*app_data.cell_size-1);
  XDrawLine(XtDisplay(w), truchet_stipple, dash_gc, 0, app_data.cell_size, 
	    2*app_data.cell_size-1, app_data.cell_size);
  XDrawArc(XtDisplay(w), truchet_stipple, dash_gc,
	   app_data.cell_size/2, 
	   -app_data.cell_size/2,
	   app_data.cell_size,
	   app_data.cell_size,
	   0,-180*64);
  XDrawArc(XtDisplay(w), truchet_stipple, dash_gc,
	   -app_data.cell_size/2, 
	   app_data.cell_size/2,
	   app_data.cell_size,
	   app_data.cell_size,
	   90*64,-180*64);
  XDrawArc(XtDisplay(w), truchet_stipple, dash_gc,
	   app_data.cell_size/2, 
	   3*app_data.cell_size/2,
	   app_data.cell_size,
	   app_data.cell_size,
	   0*64,180*64);
  XDrawArc(XtDisplay(w), truchet_stipple, dash_gc,
	   3*app_data.cell_size/2, 
	   app_data.cell_size/2,
	   app_data.cell_size,
	   app_data.cell_size,
	   90*64,180*64);
  XFreeGC(XtDisplay(w), temp_gc);


}	


/* ARGSUSED */
static void
ClearTheBigPicture()
{
  GC temp_gc;
  XGCValues values;
  XSegment segment[MAX_UNIV_WIDTH+1];
  int n_horiz_segments, n_vert_segments;
  int x, y;

  /* clear the whole thing */

  values.foreground = 1;
  values.background = 0;
  values.line_style = LineSolid; 
  values.fill_style = FillOpaqueStippled;
  temp_gc = XCreateGC(XtDisplay(canvas), big_picture,
	      GCForeground |GCBackground | GCLineStyle | GCFillStyle, &values);
  XSetStipple(XtDisplay(canvas), temp_gc, 
	      (app_data.truchet)?truchet_stipple:clear_stipple);
  XFillRectangle(XtDisplay(canvas), big_picture, temp_gc, 0, 0,
		 app_data.univ_width * app_data.cell_size,
		 app_data.univ_height * app_data.cell_size);  
  XFreeGC(XtDisplay(canvas),temp_gc);
}

static void
RefreshTheBigPicture()
{
  int	x,y;
  XExposeEvent fake_event;

  ClearTheBigPicture();
  for (x=ant.min_x; x<=ant.max_x; x++) {
    for (y=ant.min_y; y<=ant.max_y; y++) {
      if (universe[x][y] <numstates)
	DrawCellInPixmap(x,y);
    }
  }
  DrawAnt(ant.x_coord,ant.y_coord,False);
  fake_event.x = 0;
  fake_event.y = 0;
  fake_event.width = app_data.univ_width*app_data.cell_size;
  fake_event.height = app_data.univ_height*app_data.cell_size+1;
  RedrawPicture(canvas, &fake_event);
}
static void
ToggleTruchet()
{
  app_data.truchet = !app_data.truchet;
  RefreshTheBigPicture();
}

/* ARGSUSED */
static void
RedrawPicture(w, event, params, num_params)
Widget w;
XExposeEvent *event;
String *params;
Cardinal *num_params;
{
    register int x, y;
    unsigned int width, height;

    if (event) {	/* drawing because of expose or button press */
	x = event->x;
        y = event->y; 
	width = event->width;
	height =  event->height;
    } 
    else {	/* drawing because of scrolling */
	x = 0;
        y = 0; 
	width =  app_data.univ_width * app_data.cell_size; /* always the */
	height = app_data.univ_height * app_data.cell_size; /* whole window! */
    }

    if (DefaultDepthOfScreen(XtScreen(w)) == 1)
	XCopyArea(XtDisplay(w), big_picture, XtWindow(w),
		copy_gc, x, y, width, height, x, y);
    else
	XCopyPlane(XtDisplay(w), big_picture, XtWindow(w),
		copy_gc, x, y, width, height, x, y, 1);
}


/* Private Function */
static void
DrawCellInPixmap(cell_x, cell_y)
     int cell_x, cell_y;
{
  GC	*thegc;
  Boolean Hcell, Lcell;

  if (app_data.truchet) {
    XFillRectangle(XtDisplay(canvas), big_picture, /* clear the old cell */
		 draw_gc[0],
		 app_data.cell_size*cell_x,
		 app_data.cell_size*cell_y,  
		 (unsigned int)app_data.cell_size+1,
		 (unsigned int)app_data.cell_size+1);  
    XDrawRectangle(XtDisplay(canvas), big_picture, dash_gc,
		   app_data.cell_size*cell_x, app_data.cell_size*cell_y,
		   app_data.cell_size, app_data.cell_size);

    if (universe[cell_x][cell_y] < numstates)
      thegc = &canv_gc;
    else
      thegc = &dash_gc;
    Hcell =  (1+cell_x+cell_y) %2;
    Lcell = ((rule [ universe[cell_x][cell_y]%numstates ] >0)? 1: 0);
#define Vcell (!Hcell)
#define Rcell (!Lcell)

    if (app_data.debug) {
      printf("\t drawing %c-%c cell\n", 
	     Hcell?'H':'V', Lcell?'L':'R');
    }

    if ((Hcell && Rcell) || (Vcell && Lcell)) { 
     XDrawArc(XtDisplay(canvas), big_picture, *thegc,
	  (int)(app_data.cell_size*(cell_x-0.5)), /* center is the ll corner */
	  (int)(app_data.cell_size*(cell_y+0.5)),
	       (unsigned int)(app_data.cell_size),
	       (unsigned int)(app_data.cell_size),
	       90*64, -90*64);	                 /* upper right 1/4 circle  */
      XDrawArc(XtDisplay(canvas), big_picture, *thegc,
	  (int)(app_data.cell_size*(cell_x+0.5)), /* center is the ur corner */
	  (int)(app_data.cell_size*(cell_y-0.5)),
	       (unsigned int)(app_data.cell_size),
	       (unsigned int)(app_data.cell_size),
	       270*64, -90*64);			 /* lower left  1/4 circle  */
    } else {
      XDrawArc(XtDisplay(canvas), big_picture, *thegc,
	  (int)(app_data.cell_size*(cell_x-0.5)), /* center is the ul corner */
	  (int)(app_data.cell_size*(cell_y-0.5)),
	       (unsigned int)(app_data.cell_size),
	       (unsigned int)(app_data.cell_size),
	       0*64, -90*64);	                 /* lower right 1/4 circle  */
      XDrawArc(XtDisplay(canvas), big_picture, *thegc,
	  (int)(app_data.cell_size*(cell_x+0.5)), /* center is the lr corner */
	  (int)(app_data.cell_size*(cell_y+0.5)),
	       (unsigned int)(app_data.cell_size),
	       (unsigned int)(app_data.cell_size),
	       180*64, -90*64);			 /* upper left  1/4 circle  */
    }
  }

  /* Non-truchet */
  else {
    XFillRectangle(XtDisplay(canvas), big_picture, 
		 draw_gc[(universe[cell_x][cell_y])%numstates],
		 app_data.cell_size*cell_x + 1,
		 app_data.cell_size*cell_y + 1,  
		 (unsigned int)app_data.cell_size - 1,
		 (unsigned int)app_data.cell_size - 1);  
  if (universe[cell_x][cell_y] < numstates) {
    XDrawRectangle(XtDisplay(canvas), big_picture, canv_gc,
		   app_data.cell_size*cell_x,
		   app_data.cell_size*cell_y,  
		   (unsigned int)app_data.cell_size,
		   (unsigned int)app_data.cell_size);  

    }
  }
}
  
/* Private Function */
static void
DrawCell(cell_x, cell_y)
     int	cell_x, cell_y;
{
  XExposeEvent fake_event;

  DrawCellInPixmap(cell_x, cell_y);

  fake_event.x = app_data.cell_size * cell_x;
  fake_event.y = app_data.cell_size * cell_y;
  fake_event.width = app_data.cell_size+1;
  fake_event.height = app_data.cell_size+1;

  RedrawPicture(canvas, &fake_event);
}

/* Private Function */
static void
DrawAnt(cell_x, cell_y, force_event)
     int cell_x, cell_y;
     Boolean	force_event;
{
  XExposeEvent fake_event;
  XPoint opoints[4];
  XPoint ipoints[3];

  switch (ant.prevheading) {
  case 0:			/* entering going right */
    opoints[0].x = (int)app_data.cell_size*(cell_x);
    opoints[0].y = (int)app_data.cell_size*(cell_y);
    ipoints[0].x = (int)app_data.cell_size*(cell_x);
    ipoints[0].y = (int)app_data.cell_size*(cell_y+0.25);

    opoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    opoints[1].y = (int)app_data.cell_size*(cell_y+0.5);
    ipoints[1].x = (int)app_data.cell_size*(cell_x+0.25);
    ipoints[1].y = (int)app_data.cell_size*(cell_y+0.5);

    opoints[2].x = (int)app_data.cell_size*(cell_x);
    opoints[2].y = (int)app_data.cell_size*(cell_y+1);
    ipoints[2].x = (int)app_data.cell_size*(cell_x);
    ipoints[2].y = (int)app_data.cell_size*(cell_y+0.75);
    break;

  case 1:			/* entering going up */
    opoints[0].x = (int)app_data.cell_size*(cell_x);
    opoints[0].y = (int)app_data.cell_size*(cell_y+1);
    ipoints[0].x = (int)app_data.cell_size*(cell_x+0.25);
    ipoints[0].y = (int)app_data.cell_size*(cell_y+1);

    opoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    opoints[1].y = (int)app_data.cell_size*(cell_y+0.5);
    ipoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    ipoints[1].y = (int)app_data.cell_size*(cell_y+0.75);

    opoints[2].x = (int)app_data.cell_size*(cell_x+1);
    opoints[2].y = (int)app_data.cell_size*(cell_y+1);
    ipoints[2].x = (int)app_data.cell_size*(cell_x+0.75);
    ipoints[2].y = (int)app_data.cell_size*(cell_y+1);
    break;

  case 2:			/* entering going left */
    opoints[0].x = (int)app_data.cell_size*(cell_x+1);
    opoints[0].y = (int)app_data.cell_size*(cell_y+1);
    ipoints[0].x = (int)app_data.cell_size*(cell_x+1);
    ipoints[0].y = (int)app_data.cell_size*(cell_y+0.75);

    opoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    opoints[1].y = (int)app_data.cell_size*(cell_y+0.5);
    ipoints[1].x = (int)app_data.cell_size*(cell_x+0.75);
    ipoints[1].y = (int)app_data.cell_size*(cell_y+0.5);

    opoints[2].x = (int)app_data.cell_size*(cell_x+1);
    opoints[2].y = (int)app_data.cell_size*(cell_y);
    ipoints[2].x = (int)app_data.cell_size*(cell_x+1);
    ipoints[2].y = (int)app_data.cell_size*(cell_y+0.25);
    break;

  case 3:			/* entering going down */
    opoints[0].x = (int)app_data.cell_size*(cell_x);
    opoints[0].y = (int)app_data.cell_size*(cell_y);
    ipoints[0].x = (int)app_data.cell_size*(cell_x+0.25);
    ipoints[0].y = (int)app_data.cell_size*(cell_y);

    opoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    opoints[1].y = (int)app_data.cell_size*(cell_y+0.5);
    ipoints[1].x = (int)app_data.cell_size*(cell_x+0.5);
    ipoints[1].y = (int)app_data.cell_size*(cell_y+0.25);

    opoints[2].x = (int)app_data.cell_size*(cell_x+1);
    opoints[2].y = (int)app_data.cell_size*(cell_y);
    ipoints[2].x = (int)app_data.cell_size*(cell_x+0.75);
    ipoints[2].y = (int)app_data.cell_size*(cell_y);
    break;
  }

  XFillPolygon(XtDisplay(canvas), big_picture, draw_gc[0],
	       opoints, 3, Convex, CoordModeOrigin);
  XFillPolygon(XtDisplay(canvas), big_picture, canv_gc,
	       ipoints, 3, Convex, CoordModeOrigin);
  opoints[3].x = opoints[0].x;
  opoints[3].y = opoints[0].y;
  XDrawLines(XtDisplay(canvas), big_picture, canv_gc,
	       opoints, 4, CoordModeOrigin);

  if (force_event) {
    fake_event.x = app_data.cell_size * cell_x;
    fake_event.y = app_data.cell_size * cell_y;
    fake_event.width = app_data.cell_size+1;
    fake_event.height = app_data.cell_size+1;
    RedrawPicture(canvas, &fake_event);
  }

}
  
/****************************************************************************/
Initialize() {
  int 	x, y;
  int	i, j;
  static int power[11] = {1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024};

  runstat = STOP;
  curr_step_number = 0;		

  ant.x_coord = app_data.univ_width / 2;
  ant.y_coord = app_data.univ_height / 2;
  ant.heading = 3;		/* turning down */
  ant.prevheading = 0;		/* entering from right */

  if (app_data.debug) {
    printf("ant at %d,%d, heading %d, turning %d\n", 
	   ant.x_coord, ant.y_coord, ant.prevheading, ant.heading);
  }
  ant.x_prev = ant.x_coord;
  ant.y_prev = ant.y_coord;
  xtrap      = ant.x_coord;
  ytrap      = ant.y_coord;
  ant.min_x  = ant.x_coord;
  ant.max_x  = ant.x_coord;
  ant.min_y  = ant.y_coord;
  ant.max_y  = ant.y_coord;

  if (app_data.ant_number < 1 || app_data.ant_number >= 2048) {
    printf("ant rule number out of range\n");
    exit(1);
  }

  /* set up the rulestring.  This is done in a silly way, but I dont have */
  /* time to rewrite it at the moment.  it works, anyway...  */
  j = app_data.ant_number;	
  for (i = 0; i <= 10 && j >= power[i]; i++) ;
  numstates = i;
  for (i = numstates-1; i >= 0; i--) {
    if (j >= power[i]) {
      rule[numstates-i-1]=-1;
      j -= power[i];
    }
    else {
      rule[numstates-i-1]=1;
    }	
  }	

  for (x=0; x<app_data.univ_width; x++) {    /* Initialize the universe;    */ 
    for (y=0; y<app_data.univ_height; y++) { /* use numstates (instead of 0)*/ 
      universe[x][y] = numstates;	     /* to indicate that this cell  */
    }					     /* has never been visited.     */
  }

}

step_ant() {
  unsigned char	state;
  static int 	motion[4][2] = { {1, 0}, {0, -1}, {-1, 0}, {0, 1} };

  curr_step_number++;		/* increment the time */

  ant.x_prev = ant.x_coord;	/* save old values */
  ant.y_prev = ant.y_coord;	
  ant.prevheading = ant.heading;

				/* update the cell the ant is leaving */
  universe[ant.x_coord][ant.y_coord] = 
      (universe[ant.x_coord][ant.y_coord] + 1) % numstates;

				/* move the ant to the next cell */
  ant.x_coord += motion[ant.heading][0];
  ant.y_coord += motion[ant.heading][1];
  if (ant.x_coord < 0 || ant.x_coord >= app_data.univ_width || 
      ant.y_coord < 0 || ant.y_coord >= app_data.univ_height) {
		Oops();
		return;
	}
  state = universe[ant.x_coord][ant.y_coord] % numstates;
  ant.heading = ( 4 + ant.heading + rule[state] ) %4;

  if (ant.x_coord > ant.max_x) ant.max_x = ant.x_coord;
  if (ant.y_coord > ant.max_y) ant.max_y = ant.y_coord;
  if (ant.x_coord < ant.min_x) ant.min_x = ant.x_coord;
  if (ant.y_coord < ant.min_y) ant.min_y = ant.y_coord;

  if (app_data.debug) {
   printf("%d:\tant moved to %d,%d  state %d heading is %s, turning %s\n", 
	  curr_step_number,
	  ant.x_coord, ant.y_coord, state, 
	  direction[ant.prevheading], direction[ant.heading]);
 }

}


